clear all; close all; clc

%% INJECTION MODE: borehole wall temperature (qg is positive) + water outlet temperature
%% COOLING SEASON

%design parameters
%boreholes
Ny=2; % Numero di boreholes nella direzione Y
Nx=3; % Numero di boreholes nella direzione X
B=7.5; % Distanza tra i boreholes (m)
H=150;%[m]--> borehole depth
D=4;
rb=0.075;%[m] borehole radius

%ground
kg=2;
ag=1*10^-6;% m^2/s thermal diffusivity of the ground
qg=12.57;%[W/m]

%time
time_end=20*3600*365*24; %seconds in 20 years
ts=2.5*10^9;
dt = 3600;
time = (1:dt:time_end)';
n=length(time);

% Calcola la coordinata X e Y di ciascun borehole
x = 0:B:(Nx-1)*B;
y = 0:B:(Ny-1)*B;

[X, Y] = meshgrid(x, y);

C1 = [];
C2 = [];
d = 0;

% Initializing temperature values
deltaT=zeros(n,1);
T_gr=8*ones(n,1); %undisturbed ground temperature--> used for the calculation of the average borehole temperature
Tb=zeros(n,1);
T_out=zeros(n,1);

%field model
for i = 1:Ny
    for j = 1:Nx
        for z = 1:Ny
            for k = 1:Nx
                if i == z && j == k
                    d = rb^2;
                else
                    d = (X(i, j) - X(z, k))^2 + (Y(i, j) - Y(z, k))^2;
                end
                index = find(C2 == d);
                if isempty(index)
                    C2 = [C2, d];
                    C1 = [C1, 1];
                else
                    C1(index) =  C1(index) + 1;
                end
            end
        end
    end
end

ierf = @(x) (x .* erf(x)) - (1 / sqrt(pi) * (1 - exp(-x.^2)));
hc = @(s) H*s; 
dc = @(s) D*s; 
Ie = @(s) sum(C1 .* exp(-C2 .* s.^2)) / (Nx * Ny);
ILS = @(hc, dc) 2 * ierf(hc) + 2 * ierf(hc + (2 * dc)) - ierf((2 * hc) + (2 * dc)) - ierf(2 * dc);
intfunc = @(s) Ie(s) .* ILS(hc(s), dc(s)) / (H.*(s.^2));


for t = 1:length(time)
    deltaT(t) = (qg/ (4 * pi * kg) * integral(intfunc, 1/sqrt(4*ag*time(t)),inf, 'ArrayValued', true));
    Tb(t)=T_gr(t)+deltaT(t);
end

% PLOT

figure(1)
semilogx(time/ts,deltaT,'linewidth',2)
grid on
xlabel('log(t/ts) [-]')
ylabel('g(t/ts,r_b/H=0.0005,D/H=0.027) [°C]')
title('g-function response for the 3x2 borehole field in 20 years')
set(gca,'fontsize',30)

figure(2)
plot(time/(3600*24*365),Tb,'linewidth',2)
grid on
xlabel('Time [years]')
ylabel('Temperature [°C]')
title('Average borehole field temperature in 20 years')
set(gca,'fontsize',16)

